<?php

namespace Database\Seeders;

use App\Models\Tournament;
use App\Models\Team;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Carbon\Carbon;

class MilimuCupTournamentSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create the Milimu Cup Tournament
        $tournament = Tournament::create([
            'name' => 'The Late Ev. Gerishom Milimu Cup Football Tournament 3rd Edition',
            'slug' => 'milimu-cup-3rd-edition',
            'description' => 'The event is purely based on giving back to the community & supporting talent, its not politically driven.',
            'location' => 'Inglomosio Primary School',
            'venue' => 'Inglomosio Primary School',
            'organizer' => 'Milimu Cup Organizing Committee',
            'type' => 'group_stage_knockout',
            'status' => 'completed',
            'start_date' => Carbon::create(2023, 12, 18),
            'end_date' => Carbon::create(2024, 1, 1),
            'registration_start' => Carbon::create(2023, 11, 1),
            'registration_end' => Carbon::create(2023, 12, 15),
            'max_participants' => 12,
            'min_participants' => 8,
            'format' => 'football',
            'team_size' => 11,
            'rounds' => 4,
            'matches_per_round' => 3,
            'entry_fee' => 0.00,
            'prize_pool' => 0.00,
            'currency' => 'KES',
            'visibility' => 'public',
            'allow_spectators' => true,
            'is_featured' => true,
            'published_at' => Carbon::create(2023, 11, 1),
            'completed_at' => Carbon::create(2024, 1, 1),
            'rules' => [
                'lateness_penalty' => 'ksh 500 for late arrival',
                'pre_match_entertainment' => '45 minutes',
                'award_ceremony' => '5:45 P.M to 6:30 P.M',
                'payment_method' => 'cheque only',
                'break_days' => ['25th December (Christmas)', '31st December']
            ],
            'settings' => [
                'groups' => [
                    'A' => ['Ingolomosio Fc', 'Shiners Fc', 'Green Forest fc (Makuche)'],
                    'B' => ['Juventus Fc', 'Mitoto Fc', 'Clevanics Fc'],
                    'C' => ['Township Fc (Arsenal)', 'Salvation Fc', 'Mukhuyu Fc'],
                    'D' => ['Kambiri Fc', 'Young City Fc', 'Juventus B Fc']
                ],
                'special_events' => [
                    'kids_matches' => '26th December - Kids below 13 years',
                    'ladies_matches' => '28th December - Ladies teams',
                    'entertainment' => 'Eating & Dance Competition',
                    'wazee_final' => '1st January - Entertainment for elders'
                ]
            ],
            'metadata' => [
                'motto' => 'IN GOD WE TRUST',
                'edition' => '3rd Edition',
                'community_focused' => true,
                'non_political' => true
            ]
        ]);

        // Create teams for Group A
        $this->createTeamsForGroup($tournament, 'A', [
            'Ingolomosio Fc',
            'Shiners Fc', 
            'Green Forest fc (Makuche)'
        ]);

        // Create teams for Group B
        $this->createTeamsForGroup($tournament, 'B', [
            'Juventus Fc',
            'Mitoto Fc',
            'Clevanics Fc'
        ]);

        // Create teams for Group C
        $this->createTeamsForGroup($tournament, 'C', [
            'Township Fc (Arsenal)',
            'Salvation Fc',
            'Mukhuyu Fc'
        ]);

        // Create teams for Group D
        $this->createTeamsForGroup($tournament, 'D', [
            'Kambiri Fc',
            'Young City Fc',
            'Juventus B Fc'
        ]);

        // Create special teams for ladies matches
        $this->createSpecialTeams($tournament, [
            'Juventus Queens',
            'Rising Starlets',
            'The Inglets',
            'The Angels'
        ]);

        $this->command->info('Milimu Cup Tournament and teams seeded successfully!');
    }

    private function createTeamsForGroup(Tournament $tournament, string $group, array $teamNames): void
    {
        foreach ($teamNames as $teamName) {
            Team::create([
                'tournament_id' => $tournament->id,
                'name' => $teamName,
                'description' => "Team from Group {$group}",
                'status' => 'registered',
                'payment_status' => 'completed',
                'payment_amount' => 0.00,
                'payment_method' => 'cheque',
                'payment_completed_at' => Carbon::create(2023, 12, 15),
                'max_players' => 20,
                'notes' => "Group {$group} team",
                'notify_match_updates' => true,
                'notify_tournament_updates' => true,
                'stats' => [
                    'group' => $group,
                    'matches_played' => 0,
                    'wins' => 0,
                    'losses' => 0,
                    'draws' => 0,
                    'goals_for' => 0,
                    'goals_against' => 0,
                    'points' => 0
                ]
            ]);
        }
    }

    private function createSpecialTeams(Tournament $tournament, array $teamNames): void
    {
        foreach ($teamNames as $teamName) {
            Team::create([
                'tournament_id' => $tournament->id,
                'name' => $teamName,
                'description' => "Special ladies team for the tournament",
                'status' => 'registered',
                'payment_status' => 'completed',
                'payment_amount' => 0.00,
                'payment_method' => 'cheque',
                'payment_completed_at' => Carbon::create(2023, 12, 15),
                'max_players' => 15,
                'notes' => "Ladies team for special matches",
                'notify_match_updates' => true,
                'notify_tournament_updates' => true,
                'stats' => [
                    'category' => 'ladies',
                    'matches_played' => 0,
                    'wins' => 0,
                    'losses' => 0,
                    'draws' => 0,
                    'goals_for' => 0,
                    'goals_against' => 0,
                    'points' => 0
                ]
            ]);
        }
    }
} 